/**
* \file: IConfiguration.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef CARPLAY_ICONFIGURATION_H
#define CARPLAY_ICONFIGURATION_H

#include <string>
#include <list>
#include <cstdint>
#include <stdint.h>

namespace adit { namespace carplay
{

typedef void* SessionId; //unique reference passed to all plug-ins

class IConfiguration
{
public:
    /**
     * Default constructor.
     */
    IConfiguration();

    /**
     * Default virtual destructor.
     */
    virtual ~IConfiguration() {};

    /**
     * Check if an item exists.
     * Do not use with multi-line items.
     *
     * \param [in] key      key name of the item
     * \param [out] result   value of the item
     *
     * \return bool         true: if available, false: if not available
     */
    virtual bool TryGetItem(const std::string& key, std::string& result) const = 0;

    /**
     * Read configuration item.
     * In case there are multiple numbered items with the same key they will be merged to on line.
     * Same as GetMultiItem merged with spaces.
     *
     * \param [in] key      key name of the item
     *
     * \return string       value of the item or "" if not found
     */
    std::string GetItem(const std::string& key) const __attribute__((deprecated));

    /**
     * Read configuration item with warning and default.
     * In case there are multiple numbered items with the same key they will be merged to on line.
     * Same as GetMultiItem merged with spaces.
     *
     * \param [in] key      key name of the item
     * \param [in] default  default value
     *
     * \return string       value of the item or defaultValue if not found (with warning)
     */
    std::string GetItem(const std::string& key, const std::string& defaultValue) const;

    /**
     * Read configuration item with more than one line.
     * All items read until next number is not available.
     * example-item-0
     * example-item-1
     *
     * \param [in] key      key name of the item without the number and the "-" before
     *
     * \return list<string> list of the items or empty if nothing found
     */
    virtual std::list<std::string> GetItems(const std::string& key) const = 0;

    /**
     * Read item as an long integer.
     * Either decimal or hexadecimal with prefix 0x.
     *
     * \param [in] key      key name of the item
     *
     * \return long         number or 0 if not found
     */
    long GetInt(const std::string& key) const __attribute__((deprecated));

    /**
     * Read item as an integer with warning and default.
     * Either decimal or hexadecimal with prefix 0x.
     *
     * \param [in] key      key name of the item
     * \param [in] default  default value
     *
     * \return long         number or defaultValue if not found (with warning)
     */
    int64_t GetNumber(const std::string& key, int64_t defaultValue) const;
};

class IDynamicConfiguration : public IConfiguration
{
public:
    /**
     * Default virtual destructor.
     */
    virtual ~IDynamicConfiguration() {};

    /**
     * Write configuration item.
     * The item will be stored only in memory and not reflected to any file.
     *
     * \param [in] key      key name of the item
     * \param [in] value    value of the item
     */
    virtual void SetItem(const std::string& key, const std::string& value) = 0;
};

} } /* namespace adit { namespace carplay */

#endif /* CARPLAY_ICONFIGURATION_H */
